/******************************************************************************
 *    T2Scripts.h
 *
 *    This file is part of Public Scripts
 *    Copyright (C) 2005-2007 Tom N Harris <telliamed@whoopdedo.org>
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *****************************************************************************/
#ifndef T2SCRIPTS_H
#define T2SCRIPTS_H

#if (_DARKGAME == 2)

#ifndef SCR_GENSCRIPTS
#include "BaseScript.h"
#include "BaseTrap.h"
#include "CommonScripts.h"

#include "scriptvars.h"

#include <lg/scrservices.h>
#include <lg/links.h>
#endif // SCR_GENSCRIPTS

/**
 * Script: Burplauncher
 * Inherits: BaseScript
 * Links: Firer
 * Messages: Slain
 */
#if !SCR_GENSCRIPTS
class cScr_SlayFirer : public cBaseScript
{
public:
	cScr_SlayFirer(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv*, ILinkQuery*, IScript*, void*);

protected:
	virtual long OnSlain(sSlayMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("Burplauncher","BaseScript",cScr_SlayFirer)
#endif // SCR_GENSCRIPTS

/**
 * Script: CollapseFloor
 * Inherits: BasePPlateScript
 * Messages: PressurePlateActive
 */
#if !SCR_GENSCRIPTS
class cScr_CollapseFloor : public cBasePPlateScript
{
public:
	cScr_CollapseFloor(const char* pszName, int iHostObjId)
		: cBasePPlateScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnPressurePlateActive(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("CollapseFloor","BaseScript",cScr_CollapseFloor)
#endif // SCR_GENSCRIPTS

/**
 * Script: Elemental
 * Inherits: BaseAIScript
 * Messages: Damage, Slain
 * Properties: Shape\Scale, Tweq\Scale
 * Parameters: health_curve
 *
 * To get the complete T1/G behavior, use SlayHaltSpeech as well.
 */
#if !SCR_GENSCRIPTS
class cScr_Elemental : public cBaseAIScript
{
public:
	cScr_Elemental(const char* pszName, int iHostObjId)
		: cBaseAIScript(pszName, iHostObjId)
	{ }

protected:
	void UpdateScale();
	void NotifyParticles(const char* pszMsg);

protected:
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnSlain(sSlayMsg*, cMultiParm&);
	virtual long OnDamage(sDamageScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("Elemental","BaseAIScript",cScr_Elemental)
#endif // SCR_GENSCRIPTS

/**
 * Script: FireElement
 * Inherits: Elemental
 * Messages: Damage, Alertness
 * Properties: Renderer\Dynamic Light, Renderer\Extra Light, Renderer\Transparency
 * Metaproperties: BigHeatSource
 */
#if !SCR_GENSCRIPTS
class cScr_FireElement : public cScr_Elemental
{
public:
	cScr_FireElement(const char* pszName, int iHostObjId)
		: cScr_Elemental(pszName, iHostObjId)
	{ }

protected:
	virtual long OnAlertness(sAIAlertnessMsg*, cMultiParm&);
	virtual long OnDamage(sDamageScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("FireElement","Elemental",cScr_FireElement)
#endif // SCR_GENSCRIPTS

/**
 * Script: FireElemSparx
 * Inherits: BaseScript
 * Messages: TurnOn, TurnOff, Mood0, Mood1, Mood2, Mood3, Die
 */
#if !SCR_GENSCRIPTS
class cScr_FireSparx : public cBaseScript
{
public:
	cScr_FireSparx(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

private:
	void SetColor(int iColor);

protected:
	virtual long OnMessage(sScrMsg*, cMultiParm&);
	virtual long OnTurnOn(sScrMsg*, cMultiParm&);
	virtual long OnTurnOff(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("FireElemSparx","BaseScript",cScr_FireSparx)
#endif // SCR_GENSCRIPTS

/**
 * Script: HotPlate
 * Inherits: BaseScript
 * Messages: PhysContactCreate, PhysContactDestroy
 */
#if !SCR_GENSCRIPTS
class cScr_HotPlate : public cBaseScript
{
public:
	cScr_HotPlate(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnEndScript(sScrMsg*, cMultiParm&);
	virtual long OnPhysContactCreate(sPhysMsg*, cMultiParm&);
	virtual long OnPhysContactDestroy(sPhysMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("HotPlate","BaseScript",cScr_HotPlate)
#endif // SCR_GENSCRIPTS

/**
 * Script: HotPlateControl
 * Inherits: BaseScript
 * Messages: TweqComplete
 * Properties: Shape\Joint Positions, Renderer\Extra Light
 * Metaproperties: HotPlateHeat
 */
#if !SCR_GENSCRIPTS
class cScr_HotPlateCtrl : public cBaseScript
{
public:
	cScr_HotPlateCtrl(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv*, ILinkQuery*, IScript*, void*);

protected:
	virtual long OnTweqComplete(sTweqMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("HotPlateControl","BaseScript",cScr_HotPlateCtrl)
#endif // SCR_GENSCRIPTS

/**
 * Script: ModelByCount
 * Inherits: BaseScript
 * Messages: Contained
 * Properties: Engine Features\Stack Count, Tweq\Models
 */
#if !SCR_GENSCRIPTS
class cScr_ModelByCount : public cBaseScript
{
public:
	cScr_ModelByCount(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId),
		  bUpdating(false)
	{ }

private:
	void UpdateModel(void);
	
	bool bUpdating;

protected:
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnContained(sContainedScrMsg*, cMultiParm&);
	virtual long OnCombine(sCombineScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("ModelByCount","BaseScript",cScr_ModelByCount)
#endif // SCR_GENSCRIPTS

/**
 * Script: SecureDoor
 * Inherits: BaseDoorScript
 * Messages: Sim, DoorOpening, DoorClose
 * Links: AIWatchObj
 * Parameters: watcher
 */
#if !SCR_GENSCRIPTS
class cScr_SecureDoor : public cBaseDoorScript
{
public:
	cScr_SecureDoor(const char* pszName, int iHostObjId)
		: cBaseDoorScript(pszName, iHostObjId)
	{ }

private:
	int GetDoorState(void);
	cAnsiStr GetWatchers(void);

protected:
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnDoorOpening(sDoorMsg*, cMultiParm&);
	virtual long OnDoorClose(sDoorMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("SecureDoor","BaseDoorScript",cScr_SecureDoor)
#endif // SCR_GENSCRIPTS

/**
 * Script: StickyVines
 * Inherits: BaseScript
 * Messages: PhysCollision
 */
#if !SCR_GENSCRIPTS
class cScr_VineShot : public cBaseScript
{
public:
	cScr_VineShot(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnEndScript(sScrMsg*, cMultiParm&);
	virtual long OnPhysCollision(sPhysMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("StickyVines","BaseScript",cScr_VineShot)
#endif // SCR_GENSCRIPTS

/**
 * Script: JunkVines
 * Inherits: BaseScript
 * Messages: Contained
 */
#if !SCR_GENSCRIPTS
class cScr_JunkVines : public cBaseScript
{
public:
	cScr_JunkVines(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnContained(sContainedScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("JunkVines","BaseScript",cScr_JunkVines)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapCreate
 * Inherits: BaseTrap
 * Links: Contains
 */
#if !SCR_GENSCRIPTS
class cScr_CreateTrap : public cBaseTrap
{
public:
	cScr_CreateTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv*, ILinkQuery*, IScript*, void*);

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapCreate","BaseTrap",cScr_CreateTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: WatchMe
 * Inherits: BaseScript
 * Messages: BeginScript
 * Links: AIWatchObj
 * Parameters: watcher
 */
#if !SCR_GENSCRIPTS
class cScr_WatchMe : public cBaseScript
{
public:
	cScr_WatchMe(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("WatchMe","BaseScript",cScr_WatchMe)
#endif // SCR_GENSCRIPTS

/**
 * Script: WindowShade
 * Inherits: BaseTrap
 * Messages: Toggle, Slain
 * Properties: AnimLight, TerrReplaceOn, TerrReplaceOff, TerrReplaceDestroy
 */
#if !SCR_GENSCRIPTS
class cScr_WindowShade : public cBaseTrap
{
public:
	cScr_WindowShade(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId),
		  SCRIPT_VAROBJ(cScr_WindowShade,m_iOnMode,iHostObjId),
		  SCRIPT_VAROBJ(cScr_WindowShade,m_iOffMode,iHostObjId),
		  SCRIPT_VAROBJ(cScr_WindowShade,m_bBroken,iHostObjId)
	{ }

private:
	script_int m_iOnMode;
	script_int m_iOffMode;
	script_int m_bBroken;

protected:
	void InitModes(void);
	void TurnLightOn(void);
	void TurnLightOff(void);
	bool IsLightOn(void);
	void Trigger(bool bTurnOn);

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnSlain(sSlayMsg*, cMultiParm&);
	virtual long OnMessage(sScrMsg*, cMultiParm&);
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("WindowShade","BaseTrap",cScr_WindowShade)
#endif // SCR_GENSCRIPTS

/**
 * Script: ControlWindowShade
 * Inherits: WindowShade
 * Messages: Sim
 * Properties: Script\Timing
 */
#if !SCR_GENSCRIPTS
class cScr_WindowShadeRandom : public cScr_WindowShade
{
public:
	cScr_WindowShadeRandom(const char* pszName, int iHostObjId)
		: cScr_WindowShade(pszName, iHostObjId),
		  SCRIPT_VAROBJ(cScr_WindowShadeRandom,m_hTimer,iHostObjId)
	{ }

private:
	script_handle<tScrTimer> m_hTimer;

protected:
	void SetAutoTimer(bool);

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnSlain(sSlayMsg*, cMultiParm&);
	virtual long OnTimer(sScrTimerMsg*, cMultiParm&);
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("ControlWindowShade","WindowShade",cScr_WindowShadeRandom)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapSlayer
 * Inherits: BaseTrap
 * Links: ControlDevice
 * Properties: Game\Damage Model\Hit Points
 * Stimuli: BashStim
 */
#if !SCR_GENSCRIPTS
class cScr_SlayTrap : public cBaseTrap
{
public:
	cScr_SlayTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv*, ILinkQuery*, IScript*, void*);

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapSlayer","BaseTrap",cScr_SlayTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapTerminator
 * Inherits: BaseTrap
 * Links: ControlDevice
 */
#if !SCR_GENSCRIPTS
class cScr_ReallyDestroy : public cBaseTrap
{
public:
	cScr_ReallyDestroy(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv*, ILinkQuery*, IScript*, void*);

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapTerminator","BaseTrap",cScr_ReallyDestroy)
#endif // SCR_GENSCRIPTS

/**
 * Script: CorpseFrobFixed
 * Inherits: BaseAIScript
 * Messages: AIModeChange, Slain, FrobWorldEnd
 * Metaproperties: FrobInert
 */
#if !SCR_GENSCRIPTS
class cScr_NewCorpseFrob : public cBaseAIScript
{
public:
	cScr_NewCorpseFrob(const char* pszName, int iHostObjId)
		: cBaseAIScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnSlain(sSlayMsg*, cMultiParm&);
	virtual long OnAIModeChange(sAIModeChangeMsg*, cMultiParm&);
	virtual long OnFrobWorldEnd(sFrobMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("CorpseFrobFixed","BaseAIScript",cScr_NewCorpseFrob)
#endif // SCR_GENSCRIPTS


/*** miss03 ***/

/**
 * Script: ReallyLocked
 * Inherits: BaseScript
 * Messages: NowLocked, NowUnlocked
 * Metaproperties: FrobInert
 */
#if !SCR_GENSCRIPTS
class cScr_LockFrobInert : public cBaseScript
{
public:
	cScr_LockFrobInert(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnNowLocked(sScrMsg*, cMultiParm&);
	virtual long OnNowUnlocked(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("ReallyLocked","BaseScript",cScr_LockFrobInert)
#endif // SCR_GENSCRIPTS

/**
 * Script: FactoryBase
 * Inherits: BaseScript
 * Messages: TweqComplete, TurnOn, TurnOff, SynchUp, Target?
 * Links: ControlDevice, ScriptParams("ErrorOutput"), FrobProxy
 * Properties: Tweq\Joints, Tweq\JointState
 * Schemas: "Event StateChange", "DirectionState ..."
 */
#if !SCR_GENSCRIPTS
class cScr_Factory : public cBaseScript
{
public:
	cScr_Factory(const char* pszName, int iHostObjId);

protected:
	// Compatible with StdLever TargState, I hope?
	enum eLeverDirection
	{
		kLeverDirectionBackward = 0,
		kLeverDirectionForward = 1
	};

	virtual void BeginGoForward(void)
	{ }
	virtual void EndGoForward(void)
	{ }
	virtual void BeginGoBackward(void)
	{ }
	virtual void EndGoBackward(void)
	{ }

	virtual eLeverDirection NextDirection(void);
	virtual void DoReport(void);
	virtual char const * ReportType(void);

	bool IsLocked(void);
	void DoTrigger(bool bTurnOn);
	void DoTrigger(bool bTurnOn, bool bReverse, const char* pszParam);
	void DoTrigger(const char* pszMsg, bool bReverse, const char* pszParam);
	void DoGoForward(void);
	void DoGoBackward(void);
	void DoSetState(eLeverDirection dir);

private:
	static long HandleReport(cScript* pScript, sScrMsg* pMsg, sMultiParm* pReply);
	static long HandleSynchUp(cScript* pScript, sScrMsg* pMsg, sMultiParm* pReply);
	static long HandleTarget(cScript* pScript, sScrMsg* pMsg, sMultiParm* pReply);

	static int TriggerIter(ILinkSrv*, ILinkQuery*, IScript*, void*);

protected:
	virtual long OnTweqComplete(sTweqMsg*, cMultiParm&);
	virtual long OnTurnOn(sScrMsg*, cMultiParm&);
	virtual long OnTurnOff(sScrMsg*, cMultiParm&);
	virtual long OnReport(sScrMsg*, cMultiParm&);
	virtual long OnSynchUp(sScrMsg*, cMultiParm&);
	virtual long OnTarget(sScrMsg*, cMultiParm&);

};
#else // SCR_GENSCRIPTS
GEN_FACTORY("FactoryBase","BaseScript",cScr_Factory)
#endif // SCR_GENSCRIPTS

/**
 * Script: FactoryLever
 * Inherits: FactoryBase
 * Messages: FrobWorldEnd
 */
#if !SCR_GENSCRIPTS
class cScr_FactoryLever : public cScr_Factory
{
public:
	cScr_FactoryLever(const char* pszName, int iHostObjId)
		: cScr_Factory(pszName, iHostObjId)
	{ }

protected:
	virtual void BeginGoForward(void);
	virtual void EndGoForward(void);
	virtual void EndGoBackward(void);
	virtual char const * ReportType(void);

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnFrobWorldEnd(sFrobMsg*, cMultiParm&);

};
#else // SCR_GENSCRIPTS
GEN_FACTORY("FactoryLever","FactoryBase",cScr_FactoryLever)
GEN_ALIAS("FactoryCauldronLever","FactoryBase",cScr_FactoryLever,Cauldron)
GEN_ALIAS("FactoryMoldLever","FactoryBase",cScr_FactoryLever,Mold)
#endif // SCR_GENSCRIPTS

/**
 * Script: FactoryMold
 * Inherits: FactoryLever
 * Messages: FrobToolEnd, Contained, SocketMe, UnsocketMe, Full, Full?
 * Links: ScriptParams("Mold"), Owns, Contains
 * Properties: Physics\Model\Attributes
 */
#if !SCR_GENSCRIPTS
class cScr_FactoryMold : public cScr_FactoryLever
{
public:
	cScr_FactoryMold(const char* pszName, int iHostObjId)
		: cScr_FactoryLever(pszName, iHostObjId),
		  SCRIPT_VAROBJ(cScr_FactoryGauge,m_iState,iHostObjId)
	{ }

protected:
	virtual void BeginGoForward(void);
	virtual void EndGoForward(void);
	virtual void BeginGoBackward(void);
	virtual void EndGoBackward(void);
	virtual char const * ReportType(void);

private:
	script_int m_iState;

	static int ProductOwnerIter(ILinkSrv*, ILinkQuery*, IScript*, void*);

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnFrobToolEnd(sFrobMsg*, cMultiParm&);
	virtual long OnFrobWorldEnd(sFrobMsg*, cMultiParm&);
	virtual long OnFrobWorldBegin(sFrobMsg*, cMultiParm&);
	virtual long OnContained(sContainedScrMsg*, cMultiParm&);
	virtual long OnMessage(sScrMsg*, cMultiParm&);

};
#else // SCR_GENSCRIPTS
GEN_FACTORY("FactoryMold","FactoryLever",cScr_FactoryMold)
#endif // SCR_GENSCRIPTS

/**
 * Script: MoldSocket
 * Inherits: BaseScript
 * Messages: SocketMe, UnsocketMe
 * Links: ScriptParams, Owns
 * Properties: Physics\Model\Attributes, Engine Features\KeySrc, Engine Features\KeyDst
 */
#if !SCR_GENSCRIPTS
class cScr_MoldSocket : public cBaseScript
{
public:
	cScr_MoldSocket(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }


protected:
	virtual long OnMessage(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("MoldSocket","BaseScript",cScr_MoldSocket)
#endif // SCR_GENSCRIPTS

/**
 * Script: FactoryCauldron
 * Inherits: FactoryBase
 * Links: ScriptParams(Synch), ScriptParams(Sparks)
 * Properties: Tweq\Models, Renderer\AnimLight
 * Schemas: cauldron_lp, lava_pour, cauldron_pivot
 */
#if !SCR_GENSCRIPTS
class cScr_FactoryCauldron : public cScr_Factory
{
public:
	cScr_FactoryCauldron(const char* pszName, int iHostObjId)
		: cScr_Factory(pszName, iHostObjId)
	{ }

protected:
	virtual void BeginGoForward(void);
	virtual void EndGoForward(void);
	virtual void BeginGoBackward(void);
	virtual void EndGoBackward(void);
	virtual char const * ReportType(void);

protected:

};
#else // SCR_GENSCRIPTS
GEN_FACTORY("FactoryCauldron","FactoryBase",cScr_FactoryCauldron)
#endif // SCR_GENSCRIPTS

/**
 * Script: FactoryGauge
 * Inherits: FactoryBase
 * Messages: Halt, Full
 * Links: ScriptParams("Mold")
 * Parameters: rates, overflow
 */
#if !SCR_GENSCRIPTS
class cScr_FactoryGauge : public cScr_Factory
{
public:
	cScr_FactoryGauge(const char* pszName, int iHostObjId)
		: cScr_Factory(pszName, iHostObjId),
		  SCRIPT_VAROBJ(cScr_FactoryGauge,m_iState,iHostObjId),
		  SCRIPT_VAROBJ(cScr_FactoryGauge,m_fMidJoint,iHostObjId)
	{ }

protected:
	virtual void BeginGoForward(void);
	virtual void EndGoForward(void);
	virtual void BeginGoBackward(void);
	virtual void EndGoBackward(void);
	virtual char const * ReportType(void);

private:
	script_int m_iState;
	script_float m_fMidJoint;

	float GetRate(int state);

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnMessage(sScrMsg*, cMultiParm&);

};
#else // SCR_GENSCRIPTS
GEN_FACTORY("FactoryGauge","FactoryBase",cScr_FactoryGauge)
#endif // SCR_GENSCRIPTS

/**
 * Script: FactoryLight
 * Inherits: BaseScript
 * Messages: TurnOn, TurnOff
 * Properties: Renderer\Render Type, Renderer\AnimLight
 * Schemas: "Event Activate"
 */
#if !SCR_GENSCRIPTS
class cScr_LightBlinker : public cBaseScript
{
public:
	cScr_LightBlinker(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId),
		  SCRIPT_VAROBJ(cScr_LightBlinker,m_hTimer,iHostObjId)
	{ }

private:
	script_handle<tScrTimer> m_hTimer;

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnTimer(sScrTimerMsg*, cMultiParm&);
	virtual long OnTurnOn(sScrMsg*, cMultiParm&);
	virtual long OnTurnOff(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("FactoryLight","BaseScript",cScr_LightBlinker)
#endif // SCR_GENSCRIPTS

/**
 * Script: FactoryWork
 * Inherits: BaseAIScript
 * Messages: ObjActResult, MotionEnd, Alertness, AIModeChange, Slain, Report
 * Links: Owns, Route, ScriptParams("WithHammer")
 * Metaproperties: M-FactoryWorker, M-WithHammer
 * Objects: TrolPt, Molds, Hammers, SmeltingCauldr
 * Parameters: frobmotion, pickupmotion, dropmotion
 */
#if !SCR_GENSCRIPTS
class cScr_FactoryWork : public cBaseAIScript
{
public:
	cScr_FactoryWork(const char* pszName, int iHostObjId)
		: cBaseAIScript(pszName, iHostObjId),
		  SCRIPT_VAROBJ(cScr_FactoryWork,m_iCurrentAction,iHostObjId),
		  SCRIPT_VAROBJ(cScr_FactoryWork,m_iWayPtDest,iHostObjId)
	{ }

private:
	script_int m_iCurrentAction;
	script_int m_iWayPtDest;

protected:
	int GetPath(void);
	int GetOwned(int iArchetype);
	int WhatControls(int iDevice);
	bool IsLeverOn(int iDevice);
	bool IsMoldFull(int iDevice);
	bool IsContained(int iHammer);
	bool IsLocked(int iDevice);

	void GotoWayPt(int iTarget, int iSpeed=1);
	void PlayMotion(const char* pszType, const char* pszDefault);
	void FrobLever(int iTarget);
	void ChangeHammerModel(bool bWithHammer);
	void PickUp(int iHammer);
	void MakeHammer(void);
	void DecideAction(void);

	enum
	{
		kWorkDecideAction = 0,
		kWorkMoldLeverOff,
		kWorkCauldLeverOn,
		kWorkCauldLeverOff,
		kWorkMoldLeverOn,
		kWorkPickUpHammer,
		kWorkDropHammer,
		kWorkKeepHammer
	};

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnSlain(sSlayMsg*, cMultiParm&);
	virtual long OnAIModeChange(sAIModeChangeMsg*, cMultiParm&);
	virtual long OnAlertness(sAIAlertnessMsg*, cMultiParm&);
	virtual long OnObjActResult(sAIObjActResultMsg*, cMultiParm&);
	virtual long OnMotionEnd(sBodyMsg*, cMultiParm&);
	virtual long OnMessage(sScrMsg* pMsg, cMultiParm& mpReply);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("FactoryWork","BaseAIScript",cScr_FactoryWork)
#endif // SCR_GENSCRIPTS

/**
 * Script: Prisoner
 * Inherits: BaseScript
 * Messages: TurnOn
 * Metaproperties: M-Escapee
 * Signals: Escape
 */
#if !SCR_GENSCRIPTS
class cScr_Prisoner : public cBaseScript
{
public:
	cScr_Prisoner(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }


protected:
	virtual long OnTurnOn(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("Prisoner","BaseScript",cScr_Prisoner)
#endif // SCR_GENSCRIPTS

/**
 * Script: ResetLockbox
 * Inherits: BaseScript
 * Messages: NowUnlocked, TweqComplete
 * Properties: Tweq\Flicker
 */
#if !SCR_GENSCRIPTS
class cScr_LockResetter : public cBaseScript
{
public:
	cScr_LockResetter(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

private:
	void StartTimer(void);

protected:
	virtual long OnNowUnlocked(sScrMsg*, cMultiParm&);
	virtual long OnTweqComplete(sTweqMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("ResetLockbox","BaseScript",cScr_LockResetter)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapSecureDoor
 * Inherits: BaseScript
 * Messages: TurnOn, TurnOff
 * Links: AIWatchObj
 * Properties: Engine Features\Suspicious
 * Parameters: watcher
 */
#if !SCR_GENSCRIPTS
class cScr_SuspiciousTrap : public cBaseScript
{
public:
	cScr_SuspiciousTrap(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

private:
	cAnsiStr GetWatchers(void);

protected:
	virtual long OnTurnOn(sScrMsg*, cMultiParm&);
	virtual long OnTurnOff(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapSecureDoor","BaseScript",cScr_SuspiciousTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrigOBBPlayerStuff
 * Inherits: BaseTrap
 * Messages: PhysEnter, PhysExit
 */
#if !SCR_GENSCRIPTS
class cScr_OBBPlayerStuff : public cBaseTrap
{
public:
	cScr_OBBPlayerStuff(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnEndScript(sScrMsg*, cMultiParm&);
	virtual long OnPhysEnter(sPhysMsg*, cMultiParm&);
	virtual long OnPhysExit(sPhysMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrigOBBPlayerStuff","BaseTrap",cScr_OBBPlayerStuff)
#endif // SCR_GENSCRIPTS

/**
 * Script: CuttyCell
 * Inherits: BaseRoomScript
 * Messages: PlayerRoomEnter, PlayerRoomExit, Presence, Absence
 */
#if !SCR_GENSCRIPTS
class cScr_ConvRoomPlayer : public cBaseRoomScript
{
public:
	cScr_ConvRoomPlayer(const char* pszName, int iHostObjId)
		: cBaseRoomScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnPlayerRoomEnter(sRoomMsg*, cMultiParm&);
	virtual long OnPlayerRoomExit(sRoomMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("CuttyCell","BaseRoomScript",cScr_ConvRoomPlayer)
#endif // SCR_GENSCRIPTS

/**
 * Script: NearCuttyCell
 * Inherits: BaseRoomScript
 * Messages: CreatureRoomEnter, CreatureRoomExit, Obituary, Intrusion, Privacy
 * Links: Population
 * Metaproperties: M-NotifyRegion
 */
#if !SCR_GENSCRIPTS
class cScr_ConvRoomOpponent : public cBaseRoomScript, protected cTrackPopulation
{
public:
	cScr_ConvRoomOpponent(const char* pszName, int iHostObjId)
		: cBaseRoomScript(pszName, iHostObjId),
		  cTrackPopulation(iHostObjId)
	{ }

private:
	bool IsOpponent(int iObj);

protected:
	virtual long OnCreatureRoomEnter(sRoomMsg*, cMultiParm&);
	virtual long OnCreatureRoomExit(sRoomMsg*, cMultiParm&);
	virtual long OnMessage(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("NearCuttyCell","BaseRoomScript",cScr_ConvRoomOpponent)
#endif // SCR_GENSCRIPTS

/**
 * Script: ConvControl
 * Inherits: BaseScript
 * Messages: TurnOn, TurnOff, NowLocked, NowUnlocked, Presence, Absence, Intrusion, Privacy, ConvNext, ConvEnd
 * Links: ControlDevice
 */
#if !SCR_GENSCRIPTS
class cScr_ConvControl : public cBaseScript
{
public:
	cScr_ConvControl(const char* pszName, int iHostObjId);

private:
	script_int m_active;
	script_int m_busy;
	script_int m_intruded;

	void KillPrevLink(void);
	void RunNextLink(void);

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnTurnOn(sScrMsg*, cMultiParm&);
	virtual long OnTurnOff(sScrMsg*, cMultiParm&);
	virtual long OnNowLocked(sScrMsg*, cMultiParm&);
	virtual long OnNowUnlocked(sScrMsg*, cMultiParm&);
	virtual long OnConvNext(sScrMsg*, cMultiParm&);
	virtual long OnConvEnd(sScrMsg*, cMultiParm&);

private:
	static long HandlePresence(cScript* pScript, sScrMsg* pMsg, sMultiParm* pReply);
	static long HandleAbsence(cScript* pScript, sScrMsg* pMsg, sMultiParm* pReply);
	static long HandleIntrusion(cScript* pScript, sScrMsg* pMsg, sMultiParm* pReply);
	static long HandlePrivacy(cScript* pScript, sScrMsg* pMsg, sMultiParm* pReply);
	static long HandleConvNext(cScript* pScript, sScrMsg* pMsg, sMultiParm* pReply);
	static long HandleConvEnd(cScript* pScript, sScrMsg* pMsg, sMultiParm* pReply);

};
#else // SCR_GENSCRIPTS
GEN_FACTORY("ConvControl","BaseScript",cScr_ConvControl)
#endif // SCR_GENSCRIPTS

/**
 * Script: ConvSpeaker
 * Inherits: BaseScript
 * Messages: ConvNext, ConvEnd, ...
 */
#if !SCR_GENSCRIPTS
class cScr_ConvSpeaker : public cBaseScript
{
public:
	cScr_ConvSpeaker(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

private:
	int FindController(void);
	int GetObjList(const char* arg, int objs[24]);

	bool DoObjComplete(const char* arg);
	bool DoObjFail(const char* arg);
	bool DoObjInvalid(const char* arg);
	bool DoObjShow(const char* arg);
	bool DoObjHide(const char* arg);
	bool DoObjReplace(const char* arg1, const char* arg2);
	bool DoObjIsShown(const char* arg);
	bool DoObjIsComplete(const char* arg);
	bool DoVoiceOver(const char* arg);
	bool DoAmbient(const char* arg);
	bool DoSlay(const char* arg);
	bool DoConvNext(const char* arg);
	bool DoConvEnd(const char* arg);

protected:
	virtual long OnMessage(sScrMsg*, cMultiParm&);

};
#else // SCR_GENSCRIPTS
GEN_FACTORY("ConvSpeaker","BaseScript",cScr_ConvSpeaker)
#endif // SCR_GENSCRIPTS


/*** miss04 ***/

/**
 * Script: HeStartedIt
 * Inherits: BaseScript
 * Messages: Damage, Slain
 * Metaproperties: M-Swaying Burrick
 * Stimuli: Knockout
 */
#if !SCR_GENSCRIPTS
class cScr_PickAFight : public cBaseScript
{
public:
	cScr_PickAFight(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

private:
	void CheckCulpability(int iSource);

protected:
	virtual long OnSlain(sSlayMsg*, cMultiParm&);
	virtual long OnDamage(sDamageScrMsg*, cMultiParm&);
	virtual long OnMessage(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("HeStartedIt","BaseScript",cScr_PickAFight)
#endif // SCR_GENSCRIPTS

/**
 * Script: Horn
 * Inherits: BaseScript
 * Messages: Contained
 * Properties: AmbientHacked
 * Metaproperties: M-Swaying Burrick
 */
#if !SCR_GENSCRIPTS
class cScr_AccornOfQuintuplets : public cBaseScript
{
public:
	cScr_AccornOfQuintuplets(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnContained(sContainedScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("Horn","BaseScript",cScr_AccornOfQuintuplets)
#endif // SCR_GENSCRIPTS

/**
 * Script: MagicBones
 * Inherits: BaseScript
 * Messages: Slain, PhysCollision
 * Links: Owns
 */
#if !SCR_GENSCRIPTS
class cScr_MagicBone : public cBaseScript
{
public:
	cScr_MagicBone(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnEndScript(sScrMsg*, cMultiParm&);
	virtual long OnPhysCollision(sPhysMsg*, cMultiParm&);
	virtual long OnSlain(sSlayMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("MagicBones","BaseScript",cScr_MagicBone)
#endif // SCR_GENSCRIPTS

/**
 * Script: MagicCoffin
 * Inherits: BaseTrap
 * Messages: BoneSlain
 * Links: Owns
 */
#if !SCR_GENSCRIPTS
class cScr_MagicCoffin : public cBaseTrap
{
public:
	cScr_MagicCoffin(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnMessage(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("MagicCoffin","BaseTrap",cScr_MagicCoffin)
#endif // SCR_GENSCRIPTS


/*** miss05 ***/

/**
 * Script: TrapAIWake
 * Inherits: BaseTrap
 * Messages: TurnOn, WakeyWakey
 */
#if !SCR_GENSCRIPTS
class cScr_WakeTrap : public cBaseTrap
{
public:
	cScr_WakeTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }


protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapAIWake","BaseTrap",cScr_WakeTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: WakeableAI
 * Inherits: BaseScript
 * Messages: WakeyWakey
 * Properties: AI\State\Current mode
 */
#if !SCR_GENSCRIPTS
class cScr_WakeAI : public cBaseScript
{
public:
	cScr_WakeAI(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnMessage(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("WakeableAI","BaseScript",cScr_WakeAI)
#endif // SCR_GENSCRIPTS

/**
 * Script: CleanObjDestroy
 * Inherits: BaseTrap
 * Messages: TurnOn
 * Links: ControlDevice
 */
#if !SCR_GENSCRIPTS
class cScr_PoliteDestroyTrap : public cBaseTrap
{
public:
	cScr_PoliteDestroyTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	bool AttemptDestroy(void);

protected:
	virtual long OnTimer(sScrTimerMsg*, cMultiParm&);
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("CleanObjDestroy","BaseTrap",cScr_PoliteDestroyTrap)
#endif // SCR_GENSCRIPTS

#endif // _DARKGAME == 2

#endif // T2SCRIPTS_H
#ifdef SCR_GENSCRIPTS
#undef T2SCRIPTS_H
#endif
