/******************************************************************************
 *    PublicScripts.h
 *
 *    This file is part of Public Scripts
 *    Copyright (C) 2005-2007 Tom N Harris <telliamed@whoopdedo.org>
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *****************************************************************************/
#ifndef PUBLICSCRIPTS_H
#define PUBLICSCRIPTS_H

#if !SCR_GENSCRIPTS
#include "BaseScript.h"
#include "BaseTrap.h"
#include "CommonScripts.h"

#include "scriptvars.h"
#include "darkhook.h"

#include <lg/scrservices.h>
#include <lg/links.h>
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapMetaProp
 * Inherits: BaseTrap
 * Links: ScriptParams(S...), ScriptParams([AR]@...)
 */
#if !SCR_GENSCRIPTS
class cScr_MPTrap : public cBaseTrap
{
public:
	cScr_MPTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv* pLS, ILinkQuery* pLQ, IScript* pScript, void* pData);

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapMetaProp","BaseTrap",cScr_MPTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: ToolMP
 * Inherits: BaseScript
 * Links: ScriptParams("MeantDest"), ScriptParams(S...), ScriptParams([AR]@...)
 * Schemas: "Event Activate"
 * Parameters: effect="source"
 */
#if !SCR_GENSCRIPTS
class cScr_MPTool : public cBaseScript
{
public:
	cScr_MPTool(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv* pLS, ILinkQuery* pLQ, IScript* pScript, void* pData);

protected:
	virtual long OnFrobToolEnd(sFrobMsg* pFrobMsg, cMultiParm& mpReply);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("ToolMP","BaseScript",cScr_MPTool)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapScrMsgRelay
 * Inherits: BaseTrap
 * Links: ScriptParams(<msg>)
 * Parameters: status_index, data, data1, data2, data3
 */
#if !SCR_GENSCRIPTS
class cScr_SMRelay : public cBaseTrap
{
public:
	cScr_SMRelay(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv* pLS, ILinkQuery* pLQ, IScript* pScript, void* pData);

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapScrMsgRelay","BaseTrap",cScr_SMRelay)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapSMTrans
 * Inherits: BaseTrap
 * Links: ScriptParams(<msg>)
 */
#if !SCR_GENSCRIPTS
class cScr_SMTrans : public cBaseTrap
{
public:
	cScr_SMTrans(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv* pLS, ILinkQuery* pLQ, IScript* pScript, void* pData);

public:
	virtual long __stdcall ReceiveMessage(sScrMsg* pMsg, sMultiParm* pReply, eScrTraceAction eTrace);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapSMTrans","BaseTrap",cScr_SMTrans)
#endif // SCR_GENSCRIPTS

/**
 * Script: Forwarder
 * Inherits: BaseTrap
 * Links: ScriptParams(<msg>)
 */
#if !SCR_GENSCRIPTS
class cScr_Forwarder : public cBaseTrap
{
public:
	cScr_Forwarder(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv* pLS, ILinkQuery* pLQ, IScript* pScript, void* pData);

public:
	virtual long __stdcall ReceiveMessage(sScrMsg* pMsg, sMultiParm* pReply, eScrTraceAction eTrace);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("Forwarder","BaseTrap",cScr_Forwarder)
#endif // SCR_GENSCRIPTS

/**
 * Script: Validator
 * Inherits: BaseTrap
 * Messages: Validate
 * Links: ScriptParams(<index>)
 * Parameters: order="increment", increment=<steps>, rollover=<limit>
 */
#if !SCR_GENSCRIPTS
class cScr_Validator : public cBaseTrap
{
public:
	cScr_Validator(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId),
		  SCRIPT_VAROBJ(cScr_Validator, m_iValidateParam, iHostObjId)
	{ }

private:
	script_int m_iValidateParam;

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnMessage(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("Validator","BaseTrap",cScr_Validator)
#endif // SCR_GENSCRIPTS

/**
 * Script: LinkTemplate
 * Inherits: BaseTrap
 * Parameters: object=<recipient>, flavor=<linkflavor>, dest=<linkdest>, on_create, off_destroy, singleton
 */
#if !SCR_GENSCRIPTS
class cScr_LinkTrap : public cBaseTrap
{
public:
	cScr_LinkTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("LinkTemplate","BaseTrap",cScr_LinkTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapTeleportDelta
 * Inherits: BaseTrap
 * Messages: NetTeleport
 * Links: ControlDevice, ScriptParams("dm")
 */
#if !SCR_GENSCRIPTS
class cScr_DeltaTeleport : public cBaseTrap
{
public:
	cScr_DeltaTeleport(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	void Teleport(int iObj);

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
#if (_DARKGAME == 3)
	virtual long OnMessage(sScrMsg*, cMultiParm&);
#endif
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapTeleportDelta","BaseTrap",cScr_DeltaTeleport)
#endif // SCR_GENSCRIPTS

/**
 * Script: RandomRelay
 * Inherits: BaseTrap
 * Links: ControlDevice, ScriptParams(<weight>)
 * Parameters: reusable, preserve, weighted, rechargeable
 */
#if !SCR_GENSCRIPTS
class cScr_RelayRandom : public cBaseTrap
{
public:
	cScr_RelayRandom(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("RandomRelay","BaseTrap",cScr_RelayRandom)
GEN_ALIAS("RandomRelay1x2","BaseTrap",cScr_RelayRandom,1x2)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrigSim
 * Inherits: BaseTrap
 * Messages: Sim
 */
#if !SCR_GENSCRIPTS
class cScr_SimTrigger : public cBaseTrap
{
public:
	cScr_SimTrigger(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnTimer(sScrTimerMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrigSim","BaseTrap",cScr_SimTrigger)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapStim
 * Inherits: BaseTrap
 * Links: ControlDevice
 * Stimuli: ScriptStim
 * Parameters: stim=<stimulus>, strength=<intensity>
 */
#if !SCR_GENSCRIPTS
class cScr_StimTrap : public cBaseTrap
{
public:
	cScr_StimTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv* pLS, ILinkQuery* pLQ, IScript* pScript, void* pData);

protected:
	void StimLinks(float fScale, bool bSelf = false);

	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapStim","BaseTrap",cScr_StimTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: TimedStimulator
 * Inherits: TrapStim
 * Links: ScriptParams(<interval>)
 * Parameters: interval=<time>, stage=<start>, device
 */
#if !SCR_GENSCRIPTS
class cScr_StimRepeater : public cScr_StimTrap
{
public:
	cScr_StimRepeater(const char* pszName, int iHostObjId)
		: cScr_StimTrap(pszName, iHostObjId),
		  SCRIPT_VAROBJ(cScr_StimRepeater, m_hTimer, iHostObjId),
		  SCRIPT_VAROBJ(cScr_StimRepeater, m_iScale, iHostObjId),
		  SCRIPT_VAROBJ(cScr_StimRepeater, m_iInterval, iHostObjId)
	{ }

private:
	script_handle<tScrTimer> m_hTimer;
	script_int m_iScale;
	script_int m_iInterval;

	void GetLinkParams(int* piInterval, int* piInitial);
	void StartTimer(void);

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnTimer(sScrTimerMsg*, cMultiParm&);
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TimedStimulator","TrapStim",cScr_StimRepeater)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrigOBBSpec
 * Inherits: BaseTrap, BasePopulation
 * Messages: PhysEnter, PhysExit, Obituary
 * Links: ScriptParams("arc"), ScriptParams("cd"), Population
 * Metaproperties: M-NotifyRegion
 */
#if !SCR_GENSCRIPTS
class cScr_OBBSpec : public cBaseTrap, protected cTrackPopulation
{
public:
	cScr_OBBSpec(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId), 
		  cTrackPopulation(iHostObjId)
	{ }

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnEndScript(sScrMsg*, cMultiParm&);
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnPhysEnter(sPhysMsg*, cMultiParm&);
	virtual long OnPhysExit(sPhysMsg*, cMultiParm&);
	virtual long OnMessage(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrigOBBSpec","BaseTrap",cScr_OBBSpec)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrigBodyPickup
 * Inherits: BaseScript
 * Messages: Contained
 */
#if !SCR_GENSCRIPTS
class cScr_CorpseTrigger : public cBaseScript
{
public:
	cScr_CorpseTrigger(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnContained(sContainedScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrigBodyPickup","BaseScript",cScr_CorpseTrigger)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrigDamageRTC
 * Inherits: BaseScript
 * Messages: Damage
 */
#if !SCR_GENSCRIPTS
class cScr_DamageRTC : public cBaseScript
{
public:
	cScr_DamageRTC(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnDamage(sDamageScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrigDamageRTC","BaseScript",cScr_DamageRTC)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapHP
 * Inherits: BaseTrap
 * Parameters: hitcount=<n>, target=<obj>
 */
#if !SCR_GENSCRIPTS
class cScr_HPTrap : public cBaseTrap
{
public:
	cScr_HPTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapHP","BaseTrap",cScr_HPTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: AIItemGiver
 * Inherits: BaseScript
 * Messages: TransferItem
 */
#if !SCR_GENSCRIPTS
class cScr_TransferItem : public cBaseScript
{
public:
	cScr_TransferItem(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnMessage(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("AIItemGiver","BaseScript",cScr_TransferItem)
#endif // SCR_GENSCRIPTS

/**
 * Script: IntrinsicText
 * Inherits: BaseTrap
 * Parameters: text, clr, clr_red, clr_green, clr_blue
 */
#if !SCR_GENSCRIPTS
class cScr_QuickText : public cBaseTrap
{
public:
	cScr_QuickText(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	void ScanText(char* psz);
	void DisplayText(void);

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("IntrinsicText","BaseTrap",cScr_QuickText)
#endif // SCR_GENSCRIPTS

/**
 * Script: IntrinsicPlaque
 * Inherits: BaseTrap
 * Messages: FrobWorldEnd
 * Parameters: text, clr, clr_red, clr_green, clr_blue
 */
#if !SCR_GENSCRIPTS
class cScr_FrobText : public cScr_QuickText
{
public:
	cScr_FrobText(const char* pszName, int iHostObjId)
		: cScr_QuickText(pszName, iHostObjId)
	{ }

protected:
	virtual long OnFrobWorldEnd(sFrobMsg*, cMultiParm&);
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&)
	{ return 0; }
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("IntrinsicPlaque","IntrinsicText",cScr_FrobText)
#endif // SCR_GENSCRIPTS

/**
 * Script: IntrinsicCover
 * Inherits: IntrinsicText
 * Messages: WorldFocus
 */
#if !SCR_GENSCRIPTS
class cScr_FocusText : public cScr_QuickText
{
public:
	cScr_FocusText(const char* pszName, int iHostObjId)
		: cScr_QuickText(pszName, iHostObjId)
	{ }

protected:
	virtual long OnWorldSelect(sScrMsg*, cMultiParm&);
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&)
	{ return 0; }
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("IntrinsicCover","IntrinsicText",cScr_FocusText)
#endif // SCR_GENSCRIPTS

/**
 * Script: InvSchema
 * Inherits: BaseScript
 * Messages: FrobInvEnd
 * Links: SoundDescription
 */
#if !SCR_GENSCRIPTS
class cScr_InvSchema : public cBaseScript
{
public:
	cScr_InvSchema(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnFrobInvEnd(sFrobMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("InvSchema","BaseScript",cScr_InvSchema)
#endif // SCR_GENSCRIPTS

/**
 * Script: Sippable
 * Inherits: Script
 * Messages: FrobInvEnd, Contained
 * Properties: Inventory\Object Name, Obj\Object name
 * Parameters: sips=<n>
 */
#if !SCR_GENSCRIPTS
class cScr_Sippable : public cBaseScript
{
public:
	cScr_Sippable(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

private:
	void SetSipsLeft(int iSips);
	int GetSipsLeft(void);

protected:
	virtual long OnFrobInvEnd(sFrobMsg*, cMultiParm&);
	virtual long OnContained(sContainedScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("Sippable","BaseScript",cScr_Sippable)
#endif // SCR_GENSCRIPTS

/**
 * Script: Zapper
 * Inherits: BaseAIScript
 * Messages: EndAttack
 * Links: AITarget
 * Stimuli: MagicZapStim, Anti-Human
 * Schemas: ghmagic, hit_magic, dam_sting, exp_cryopsi
 * Parameters: zap_strength, no_zap_sound
 */
#if !SCR_GENSCRIPTS
class cScr_Zapper : public cBaseAIScript
{
public:
	cScr_Zapper(const char* pszName, int iHostObjId)
		: cBaseAIScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnEndAttack(sAttackMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("Zapper","BaseAIScript",cScr_Zapper)
#endif // SCR_GENSCRIPTS

/**
 * Script: PhysModelCorrect
 * Inherits: BaseScript
 * Messages: Sim, PMResize
 * Properties: Physics\Model, Shape\Scale
 */
#if !SCR_GENSCRIPTS
class cScr_PhysScale : public cBaseScript
{
public:
	cScr_PhysScale(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

private:
	void DoResize(void);

protected:
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnMessage(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("PhysModelCorrect","BaseScript",cScr_PhysScale)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapPhantom
 * Inherits: BaseTrap
 * Links: ControlDevice
 * Properties: Renderer\Transparency
 * Parameters: alpha_min, alpha_max, rate, curve
 */
#if !SCR_GENSCRIPTS
class cScr_AlphaTrap : public cBaseTrap
{
public:
	cScr_AlphaTrap(const char* pszName, int iHostObjId);

private:
	script_handle<tScrTimer> m_hFadeTimer;
	script_int m_iSign;
	script_int m_iStartTime;

	static const int sm_iFadeResolution;

	float m_fAlphaMin, m_fAlphaMax;
	int m_iFadeTime, m_iCurve;

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnTimer(sScrTimerMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapPhantom","BaseTrap",cScr_AlphaTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrigAIAwareShift
 * Inherits: BaseScript
 * Links: ScriptParams(af...), ScriptParams(av...), ScriptParams(aa...)
 * Hooks: Link(AIAwareness)
 */
#if !SCR_GENSCRIPTS
class cScr_TrackAwareness : public cBaseScript
{
public:
	cScr_TrackAwareness(const char* pszName, int iHostObjId);

private:
	script_int m_iPrevVis;
	script_int m_iPrevAud;

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnEndScript(sScrMsg*, cMultiParm&);
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnNotify(sDHNotifyMsg*, cMultiParm&);

private:
	static long HandleDHNotify(cScript* pScript, sScrMsg* pMsg, sMultiParm* pReply);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrigAIAwareShift","BaseScript",cScr_TrackAwareness)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapMotion
 * Inherits: BaseTrap
 * Parameters: mot
 */
#if !SCR_GENSCRIPTS
class cScr_MotionTrap : public cBaseTrap
{
public:
	cScr_MotionTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapMotion","BaseTrap",cScr_MotionTrap)
#endif // SCR_GENSCRIPTS

#if (_DARKGAME == 2) || (_DARKGAME == 3)
/**
 * Script: TrapCamShift
 * Inherits: BaseTrap
 */
#if !SCR_GENSCRIPTS
class cScr_CameraTrap : public cBaseTrap
{
public:
	cScr_CameraTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapCamShift","BaseTrap",cScr_CameraTrap)
#endif // SCR_GENSCRIPTS
#endif // (_DARKGAME == 2) || (_DARKGAME == 3)


/**
 * Script: Spy
 * Inherits: 
 */
#if !SCR_GENSCRIPTS
class cScr_Spy : public cScript
{
public:
	cScr_Spy(const char* pszName, int iHostObjId)
		: cScript(pszName, iHostObjId)
	{ }

	virtual long __stdcall ReceiveMessage(sScrMsg* pMsg, sMultiParm* pReply, eScrTraceAction eTrace);

private:
	cAnsiStr FormatMultiParm(const sMultiParm& mp, const char* pszExtra);
	cAnsiStr FormatObject(int iObjId);
	inline const char* ContainsEvent(int event);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("Spy","CustomScript",cScr_Spy)
#endif // SCR_GENSCRIPTS


/**
 * Script: TrapTimer
 * Inherits: BaseTrap
 * Properties: Script\Timing
 */
#if !SCR_GENSCRIPTS
class cScr_Delayer : public cBaseTrap
{
public:
	cScr_Delayer(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	void DelayMessage(const char* pszMsg);

protected:
	virtual long OnTimer(sScrTimerMsg*, cMultiParm&);
	virtual long OnTurnOn(sScrMsg*, cMultiParm&);
	virtual long OnTurnOff(sScrMsg*, cMultiParm&);
	virtual long OnMessage(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapTimer","BaseTrap",cScr_Delayer)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrigFlipFlop
 * Inherits: BaseTrap
 * Properties: Script\Timing
 */
#if !SCR_GENSCRIPTS
class cScr_FlipFlop : public cBaseTrap
{
public:
	cScr_FlipFlop(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId),
		  SCRIPT_VAROBJ(cScr_FlipFlop,m_iState,iHostObjId)
	{ }

private:
	script_int m_iState;

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnTimer(sScrTimerMsg*, cMultiParm&);
	virtual long OnTurnOn(sScrMsg*, cMultiParm&);
	virtual long OnTurnOff(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapFlipFlop","BaseTrap",cScr_FlipFlop)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapCinema
 * Inherits: BaseTrap
 * Parameters: movie
 */
#if !SCR_GENSCRIPTS
class cScr_PlayMovie : public cBaseTrap
{
public:
	cScr_PlayMovie(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapCinema","BaseTrap",cScr_PlayMovie)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapQVarText
 * Inherits: BaseTrap, BaseQVarText
 */
#if !SCR_GENSCRIPTS
class cScr_TrapQVText : public cBaseTrap, protected cQVarText
{
public:
	cScr_TrapQVText(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId), cQVarText(iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapQVarText","BaseTrap",cScr_TrapQVText)
#endif // SCR_GENSCRIPTS

/**
 * Script: QVarPlaque
 * Inherits: BaseScript, BaseQVarText
 * Messages: FrobWorldEnd
 */
#if !SCR_GENSCRIPTS
class cScr_FrobQVText : public cBaseScript, protected cQVarText
{
public:
	cScr_FrobQVText(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId), cQVarText(iHostObjId)
	{ }

protected:
	virtual long OnFrobWorldEnd(sFrobMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("QVarPlaque","BaseScript",cScr_FrobQVText)
#endif // SCR_GENSCRIPTS

/**
 * Script: QVarScroll
 * Inherits: BaseScript, BaseQVarText
 * Messages: FrobInvEnd
 */
#if !SCR_GENSCRIPTS
class cScr_InvQVText : public cBaseScript, protected cQVarText
{
public:
	cScr_InvQVText(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId), cQVarText(iHostObjId)
	{ }

protected:
	virtual long OnFrobInvEnd(sFrobMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("QVarScroll","BaseScript",cScr_InvQVText)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapMissionQVar
 * Inherits: BaseTrap, BaseQuestVariable
 * Properties: Trap\Quest Var, Script\QBName
 * Parameters: initqv
 */
#if !SCR_GENSCRIPTS
class cScr_TrapQVarMis : public cBaseTrap, protected cQVarProcessor
{
public:
	cScr_TrapQVarMis(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSim(sSimMsg*, cMultiParm&);
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);

	virtual int GetQVar(const char* pszName)
		{ return cQVarProcessor::GetQVar(ObjId(), pszName); }
	virtual void SetQVar(const char* pszName, int iVal)
		{ cQVarProcessor::SetQVar(ObjId(), pszName, iVal, false); }
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapMissionQVar","BaseTrap",cScr_TrapQVarMis)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapCampaignQVar
 * Inherits: TrapMissionQVar
 * Properties: Trap\Quest Var, Script\QBName
 */
#if !SCR_GENSCRIPTS
class cScr_TrapQVarCmp : public cScr_TrapQVarMis
{
public:
	cScr_TrapQVarCmp(const char* pszName, int iHostObjId)
		: cScr_TrapQVarMis(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSim(sSimMsg*, cMultiParm&);

	virtual void SetQVar(const char* pszName, int iVal)
		{ cQVarProcessor::SetQVar(ObjId(), pszName, iVal, true); }
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapCampaignQVar","TrapMissionQVar",cScr_TrapQVarCmp)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrigQuestVar
 * Inherits: BaseTrap, BaseQuestVariable
 * Properties: Trap\Quest Var, Script\QBName
 */
#if !SCR_GENSCRIPTS
class cScr_TrigQVar : public cBaseTrap, protected cQVarProcessor
{
public:
	cScr_TrigQVar(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnBeginScript(sScrMsg*, cMultiParm&);
	virtual long OnEndScript(sScrMsg*, cMultiParm&);
	virtual long OnQuestChange(sQuestMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrigQuestVar","BaseTrap",cScr_TrigQVar)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapQVarFilter
 * Inherits: BaseTrap, BaseQuestVariable
 * Properties: Trap\Quest Var, Script\QBName
 */
#if !SCR_GENSCRIPTS
class cScr_TrapQVarRelay : public cBaseTrap, protected cQVarProcessor
{
public:
	cScr_TrapQVarRelay(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapQVarFilter","BaseTrap",cScr_TrapQVarRelay)
#endif // SCR_GENSCRIPTS

/**
 * Script: BasePostRead
 * Inherits: BaseScript, BaseQuestVariable
 * Links: SoundDescription
 * Properties: Book\Text, Book\Art, Script\Use Msg, Trap\Quest Var, Script\QBName, Script\Timing, Script\Delay Time, Sound\Object Sound
 * 
 * Display text then perform some action after it has been displayed.
 * If there is a SoundDescription link, then it will play the schema
 * as a voice-over. If there is a Trap\Quest Var, then the variable is
 * set (which may occur after the schema has played).
 */
#if !SCR_GENSCRIPTS
class cScr_PostReader : public cBaseTrap, protected cQVarProcessor
{
public:
	cScr_PostReader(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	bool DoVoiceOver(int iFrobber);
	bool DoQVar(void);

protected:
	void Read(int iFrobber);

	virtual long OnTimer(sScrTimerMsg*, cMultiParm&);
	virtual long OnTurnOn(sScrMsg*, cMultiParm&);
	virtual long OnTurnOff(sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("BasePostRead","BaseTrap",cScr_PostReader)
#endif // SCR_GENSCRIPTS

/**
 * Script: PostReadPlaque
 * Inherits: BasePostRead
 * Messages: FrobWorldEnd
 */
#if !SCR_GENSCRIPTS
class cScr_FrobPostRead : public cScr_PostReader
{
public:
	cScr_FrobPostRead(const char* pszName, int iHostObjId)
		: cScr_PostReader(pszName, iHostObjId)
	{ }

protected:
	virtual long OnFrobWorldEnd(sFrobMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("PostReadPlaque","BasePostRead",cScr_FrobPostRead)
#endif // SCR_GENSCRIPTS

/**
 * Script: PostReadScroll
 * Inherits: BasePostRead
 * Messages: FrobInvEnd
 */
#if !SCR_GENSCRIPTS
class cScr_InvPostRead : public cScr_PostReader
{
public:
	cScr_InvPostRead(const char* pszName, int iHostObjId)
		: cScr_PostReader(pszName, iHostObjId)
	{ }

protected:
	virtual long OnFrobInvEnd(sFrobMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("PostReadScroll","BasePostRead",cScr_InvPostRead)
#endif // SCR_GENSCRIPTS


/**
 * Script: TrigWaypoint
 * Inherits: BaseMovingTerrainScript
 * Messages: WaypointReached
 */
#if !SCR_GENSCRIPTS
class cScr_TrigTerr : public cBaseMovingTerrainScript
{
public:
	cScr_TrigTerr(const char* pszName, int iHostObjId)
		: cBaseMovingTerrainScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnWaypointReached(sWaypointMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrigWaypoint","BaseMovingTerrainScript",cScr_TrigTerr)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapDestroyDoor
 * Inherits: BaseTrap
 * Messages: TurnOn
 * Links: ControlDevice
 * Properties: Door\Rotating, Door\Translating
 */
#if !SCR_GENSCRIPTS
class cScr_DoorKillerTrap : public cBaseTrap
{
public:
	cScr_DoorKillerTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

private:
	static int LinkIter(ILinkSrv*, ILinkQuery*, IScript*, void*);

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapDestroyDoor","BaseTrap",cScr_DoorKillerTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrigOBBCreature
 * Inherits: TrigOBBSpec
 * Messages: PhysEnter, PhysExit, Obituary
 * Links: Population
 * Metaproperties: M-NotifyRegion
 */
#if !SCR_GENSCRIPTS
class cScr_OBBCret : public cScr_OBBSpec
{
public:
	cScr_OBBCret(const char* pszName, int iHostObjId)
		: cScr_OBBSpec(pszName, iHostObjId)
	{ }

protected:
	virtual long OnPhysEnter(sPhysMsg*, cMultiParm&);
	virtual long OnPhysExit(sPhysMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrigOBBCreature","TrigOBBSpec",cScr_OBBCret)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapFader
 * Inherits: BaseTrap
 * Messages: NetFade
 * Parameters: fade_time, fade_color
 */
#if !SCR_GENSCRIPTS
class cScr_FadeTrap : public cBaseTrap
{
public:
	cScr_FadeTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
#if (_DARKGAME == 3)
	virtual long OnMessage(sScrMsg*, cMultiParm&);
#endif
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapFader","BaseTrap",cScr_FadeTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: Transmogrify
 * Inherits: BaseScript
 * Messages: Contained
 * Links: Transmute, Mutate
 */
#if !SCR_GENSCRIPTS
class cScr_Transmogrify : public cBaseScript
{
public:
	cScr_Transmogrify(const char* pszName, int iHostObjId)
		: cBaseScript(pszName, iHostObjId)
	{ }

protected:
	virtual long OnContained(sContainedScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("Transmogrify","BaseScript",cScr_Transmogrify)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapRequireOne
 * Inherits: BaseTrap, BaseRequirement
 */
#if !SCR_GENSCRIPTS
class cScr_RequireOneTrap : public cBaseTrap, protected cRequirement
{
public:
	cScr_RequireOneTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId),
		  cRequirement(iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapRequireOne","BaseTrap",cScr_RequireOneTrap)
#endif // SCR_GENSCRIPTS

/**
 * Script: TrapRequireOneOnly
 * Inherits: BaseTrap, BaseRequirement
 */
#if !SCR_GENSCRIPTS
class cScr_RequireSingleTrap : public cBaseTrap, protected cRequirement
{
public:
	cScr_RequireSingleTrap(const char* pszName, int iHostObjId)
		: cBaseTrap(pszName, iHostObjId),
		  cRequirement(iHostObjId)
	{ }

protected:
	virtual long OnSwitch(bool, sScrMsg*, cMultiParm&);
};
#else // SCR_GENSCRIPTS
GEN_FACTORY("TrapRequireOneOnly","BaseTrap",cScr_RequireSingleTrap)
#endif // SCR_GENSCRIPTS


/** TODO:
 * 
 */

#endif // PUBLICSCRIPTS_H
